library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

library UNISIM;
use UNISIM.Vcomponents.ALL;

entity exploder_bl is
	generic (
		any_in_width   : integer := 8;
		ecl_out_width  : integer := 8;
		lvds_out_width : integer := 8;
		lemo_out_width : integer := 8;
		lemo_in_width  : integer := 8;
		use_butis_receiver     : boolean := true;
		use_butis_sender       : boolean := true;
		use_rataclock_receiver : boolean := true;
		use_uart_output        : boolean := true;
		use_ratatime_receiver  : boolean := false;
		use_ratatime_sender    : boolean := true
	);
	port (
		-- Internal clock
		CK_125P_IN	: in std_logic;
		CK_125N_IN	: in std_logic;
		-- add-on board I/Os
		any_input	: in  std_logic_vector(any_in_width - 1 downto 0);
		lemo_input	: in std_logic_vector(lemo_in_width - 1 downto 0);
		ecl_output	: out std_logic_vector(ecl_out_width - 1 downto 0);
		lvds_output	: out std_logic_vector(lvds_out_width - 1 downto 0);
		lemo_output	: out std_logic_vector(lemo_out_width - 1 downto 0);
		-- config jumpers
		res	: in std_logic_vector(6 downto 0);
		res_out : out std_logic;
		-- NIM/TTL select
		select_out	: out std_logic_vector(1 downto 0);
		-- LEDs outputs.
		-- LED colors:
		-- 0 = red, blue, green, white, red, blue, green, white
		hpv			: out std_logic_vector(7 downto 0);
		-- Display background light
		-- 0 = red, green, blue
		display_led_out	: out std_logic_vector(2 downto 0)
	);
end;

-- ----------------------------------------------------------------------------

-- Current clock sources and relationships:
--
-- Board oscillator pad
-- +- clk_125p_i/clk_125n_i
--    +- PLL
--       +- clk_125_int
--       +- clk_100
--       +- clk_200_int
--          +- LVDS_OUT(0) (as clk_200_to_lvds)
--
-- LEMO(0)
-- +- rataclock_in
--    +- PLL
--       +- clk_125_rata
--       +- clk_125_90_rata
--
-- ANY_IN(0)
-- +- clk_200_butis_buf
--    +- PLL
--       +- clk_200_butis
--          +- clk_200 = recv_butis_clock  (aliases)
--       +- clk_125_butis
--       +- clk_10_butis
--
-- clk_125_int
-- clk_125_rata
-- clk_125_butis
-- +- CLK_MUX
--    +- clk_125

-- ----------------------------------------------------------------------------

-- Suggestion for clock sources and relationships:
--
-- Board oscillator pad
-- +- clk_125p_i/clk_125n_i
--    +- PLL
--       +- clk_125_int    (not used)
--       +- clk_200_int
--
-- LEMO(0)
-- +- rataclock_in
--    +- PLL
--       +- clk_200_rata
--
-- ANY_IN(0)
-- +- clk_200_butis_buf
--    +- PLL
--       +- clk_200_butis
--
-- clk_200_int
-- clk_200_rata
-- clk_200_butis
-- +- CLK_MUX
--    +- clk_200_mux
--       +- PLL
--          +- clk_200
--          +- clk_200_90
--          +- clk_100
--          +- LVDS_OUT(0) (as clk_200_to_lvds)
--
-- Try to run the receiver logics on the clk_200 / clk_200_90.
-- That would mean that e.g. the rataclock message can be decoded
-- based on the butis clock.
-- Also means that internal data need not cross clock domains.
--
-- The clk_100 for ratatime sender, in case it is unable to run at 100 MHz.
-- The ratatime receiver is not very useful, except to check if signal
-- is lockable.  That most likely can only run at 100 MHz...

-- ----------------------------------------------------------------------------

architecture behavior of exploder_bl is

	-- clocks
	signal clk_100		     : std_logic	:= '0';
	signal clk_200		     : std_logic	:= '0';
	signal clk_200_int       : std_logic	:= '0';
	signal clk_125		     : std_logic	:= '0';
	signal clk_125_90        : std_logic	:= '0';
	signal clk_125_int       : std_logic	:= '0';
	signal clk_125_mux       : std_logic	:= '0';
	signal clk_125_rata	     : std_logic	:= '0';
	signal clk_10_butis      : std_logic	:= '0';
	signal clk_125_butis     : std_logic	:= '0';
	signal clk_200_butis     : std_logic	:= '0';
	signal clk_200_butis_buf : std_logic	:= '0';
	signal clk_200_to_lvds   : std_logic	:= '0';

	signal clk_select : std_logic_vector(1 downto 0) := "00";

	-- PLL locks
	signal locked_25        : std_logic := '0';
	signal locked_100       : std_logic := '0';
	signal locked_125       : std_logic := '0';
	signal locked_200_butis : std_logic := '0';

	-- Input clock stopped?
	signal stopped_200_butis : std_logic := '0';

	-- config
	signal res_pp0	: std_logic_vector(6 downto 0) := (others => '0');
	signal res_pp1	: std_logic_vector(6 downto 0) := (others => '0');
	signal res_pp2	: std_logic_vector(6 downto 0) := (others => '0');

	-- time
	signal tick_ns			: unsigned(63 downto 0) := (others => '0');
	signal tick_ns_pp1		: unsigned(63 downto 0) := (others => '0');
	signal tick_ns_pp2		: unsigned(63 downto 0) := (others => '0');
	signal tick_ns_recv	    : unsigned(63 downto 0) := (others => '0');
	signal tick_ns_recv_pp1	: unsigned(63 downto 0) := (others => '0');
	signal tick_ns_recv_pp2	: unsigned(63 downto 0) := (others => '0');
	signal tick_ns_recv_pp3	: unsigned(63 downto 0) := (others => '0');
	signal tick_ns_recv_pp4	: unsigned(63 downto 0) := (others => '0');
	signal timestamp_ns_xx2	: unsigned(63 downto 0) := (others => '0');
	signal timestamp_ns_xx1	: unsigned(63 downto 0) := (others => '0');
	signal timestamp_ns	: unsigned(63 downto 0) := (others => '0');
	signal timestamp_ns_100	: unsigned(63 downto 0) := (others => '0');

	-- for rataser_clock_send
	signal send_clk		: std_logic := '0';
	signal send_aux_sigs	: std_logic_vector(4 downto 0) := (others => '0');
	signal send_info_bit	: std_logic := '0';
	signal send_rataclock_xx1		: std_logic := '0';
	signal send_rataclock_lvds_out	: std_logic := '0';
	signal send_rataclock_lemo_out	: std_logic := '0';
	signal send_rataclock_ecl_out	: std_logic := '0';
	signal send_rataclock_lvds_xx2	: std_logic := '0';
	signal send_rataclock_lvds_xx3	: std_logic := '0';
	signal send_rataclock_lvds_xx4	: std_logic := '0';
	signal send_rataclock_lvds_xx5	: std_logic := '0';
	signal send_rataclock_lemo_xx2	: std_logic := '0';
	signal send_rataclock_lemo_xx3	: std_logic := '0';
	signal send_rataclock_lemo_xx4	: std_logic := '0';
	signal send_rataclock_lemo_xx5	: std_logic := '0';
	signal send_rataclock_lemo_xx6	: std_logic := '0';
	signal send_rataclock_lemo_xx7	: std_logic := '0';
	signal send_rataclock_lemo_xx8	: std_logic := '0';
	signal send_rataclock_ecl_xx2	: std_logic := '0';
	signal send_rataclock_ecl_xx3	: std_logic := '0';
	signal send_rataclock_ecl_xx4	: std_logic := '0';
	signal send_rataclock_ecl_xx5	: std_logic := '0';
	signal send_rataclock_ecl_xx6	: std_logic := '0';
	signal send_rataclock_ecl_xx7	: std_logic := '0';
	signal send_rataclock_ecl_xx8	: std_logic := '0';
	signal send_rataclock_ecl_xx9	: std_logic := '0';
	signal send_rataclock_ecl_xxA	: std_logic := '0';
	signal send_rataclock_slow	: std_logic := '0';
	signal send_rataclock_oddr_lemo_naive	: std_logic := '0';
	signal send_rataclock_oddr_lemo_trail	: std_logic := '0';
	signal send_rataclock_oddr_lvds_naive	: std_logic := '0';
	signal send_rataclock_oddr_ecl	: std_logic := '0';
	signal send_rataclock_sync	    : std_logic := '0';
	signal send_rataclock_slow_sync	: std_logic := '0';
	signal send_trail_long		: std_logic := '0';
	signal send_trail_short		: std_logic := '0';
	signal send_low_min_clks : std_logic_vector(5 downto 0) := (others => '0');
	signal send_low_max_clks : std_logic_vector(5 downto 0) := (others => '0');
	signal d0_prepare : std_logic := '0';
	signal d1_prepare : std_logic := '0';


	-- for rataser_clock_recv
	signal recv_clk   : std_logic := '0';
 	signal recv_clk90 : std_logic := '0';
	signal recv_rataclock  : std_logic := '0';
	signal recv_tick_ns_lo : std_logic_vector(31 downto  0) := (others => '0');
	signal recv_tick_ns_hi : std_logic_vector(63 downto 32) := (others => '0');
	signal recv_aux_sigs   : std_logic_vector(4 downto 0) := (others => '0');
	signal recv_info_bit   : std_logic := '0';
	signal recv_msg_strobe : std_logic := '0';
	signal recv_expect_edge     : std_logic_vector(1 downto 0) := "11";
	signal recv_last_edges      : std_logic_vector(7 downto 0);
	signal recv_auto_edge       : std_logic_vector(1 downto 0);
	signal recv_sync_status_out : std_logic_vector(2 downto 0);
	signal recv_sync_status_xx1 : std_logic_vector(2 downto 0);
	signal recv_sync_status_xx2 : std_logic_vector(2 downto 0);
	signal recv_sync_status_xx3 : std_logic_vector(2 downto 0);
	signal recv_sync_lost_out   : std_logic_vector(2 downto 0);
	signal recv_sync_lost_xx1   : std_logic_vector(2 downto 0);
	signal recv_sync_lost_xx2   : std_logic_vector(2 downto 0);
	signal recv_sync_lost_xx3   : std_logic_vector(2 downto 0);
	-- unused signal recv_bad_signals     : std_logic_vector(4 downto 0);
	signal recv_clear_status    : std_logic := '0';

	signal rataclock_in    : std_logic := '0';

	-- for butis receiver
	signal recv_butis_clock      : std_logic := '0';
	signal butis_count          : unsigned(31 downto 0) := (others => '0');
	signal butis_time           : std_logic := '0';
	signal butis_time_in        : std_logic := '0';
	signal recv_butis_t0        : std_logic := '0';
    signal recv_butis_timestamp : std_logic_vector(63 downto 0) := (others => '0');
    signal recv_butis_timestamp_pp1 : std_logic_vector(63 downto 0) := (others => '0');
    signal recv_butis_timestamp_pp2 : std_logic_vector(63 downto 0) := (others => '0');
    signal recv_butis_timestamp_lo : std_logic_vector(31 downto 0) := (others => '0');
    signal recv_butis_timestamp_hi : std_logic_vector(31 downto 0) := (others => '0');
    signal recv_butis_synced : std_logic := '0';
    signal recv_butis_error  : std_logic := '0';

	-- for ratatime receiver
	signal recv_ratatime_timestamp : std_logic_vector(63 downto 0) := (others => '0');
	signal recv_ratatime_timestamp_pp1 : std_logic_vector(63 downto 0) := (others => '0');
	signal recv_ratatime_timestamp_pp2 : std_logic_vector(63 downto 0) := (others => '0');

	-- config
	signal select_200_mhz_clk				: std_logic := '0';
	signal select_rataclock_duty			: std_logic := '0';
	signal select_125_mhz_clk 				: std_logic_vector(1 downto 0) := (others => '0');
	signal select_timestamp_source		: std_logic_vector(1 downto 0) := (others => '0');
	signal select_led_display				: std_logic_vector(1 downto 0) := "00";

	-- for butis sender
	signal send_butis_t0_signal : std_logic := '0';
	signal send_butis_timestamp : std_logic_vector(63 downto 0) := (others => '0');
	signal send_butis_timestamp_next_t0 : std_logic_vector(63 downto 0) := (others => '0');
	signal send_butis_timestamp_next_t0_pp1 : std_logic_vector(63 downto 0) := (others => '0');
	signal send_butis_timestamp_next_t0_pp2 : std_logic_vector(63 downto 0) := (others => '0');
	signal send_butis_time_xx6   : std_logic := '0';
	signal send_butis_time_xx5   : std_logic := '0';
	signal send_butis_time_xx4   : std_logic := '0';
	signal send_butis_time_xx3   : std_logic := '0';
	signal send_butis_time_xx2   : std_logic := '0';
	signal send_butis_time_xx1   : std_logic := '0';
	signal send_butis_time       : std_logic := '0';
	signal send_butis_time_oddr_lvds : std_logic := '0';
	signal send_butis_mark100kHz : std_logic := '0';
	signal send_butis_cycles     : integer range 0 to 2000 := 2000;

	-- for ratatime sender
	signal send_ratatime : std_logic := '0';
	signal send_ratatime_oddr_lemo	: std_logic := '0';
	signal send_ratatime_sync : std_logic := '0';

	-- for ratatime receiver
	signal clk_100_counter : std_logic_vector(13 downto 0) := (others => '0');
	signal clk_100_next_counter : std_logic_vector(13 downto 0);
	signal recv_ratatime : std_logic := '0';
	signal recv_ratatime_new_cycle_pulse : std_logic := '0';
	signal recv_ratatime_clear : std_logic := '0';
	signal recv_ratatime_sync : std_logic_vector(2 downto 0) := (others => '0');
	signal recv_ratatime_sync_lost : std_logic_vector(1 downto 0) := (others => '0');
	signal recv_ratatime_bad_bits : std_logic_vector(3 downto 0) := (others => '0');
	signal recv_ratatime_ticks : std_logic_vector(63 downto 0);
	signal recv_ratatime_aux_bits : std_logic_vector(14 downto 0);
	signal recv_ratatime_aux_sigs : std_logic_vector(6 downto 0);
	signal recv_ratatime_speed : std_logic_vector(1 downto 0);

	signal recv_ratatime_clear_100 : std_logic := '0';
	signal recv_ratatime_sync_100 : std_logic_vector(2 downto 0) := (others => '0');
	signal recv_ratatime_sync_lost_100 : std_logic_vector(1 downto 0) := (others => '0');
	signal recv_ratatime_bad_bits_100 : std_logic_vector(3 downto 0) := (others => '0');
	signal recv_ratatime_ticks_100 : std_logic_vector(63 downto 0);
	signal recv_ratatime_aux_bits_100 : std_logic_vector(14 downto 0);
	signal recv_ratatime_aux_sigs_100 : std_logic_vector(6 downto 0);
	signal recv_ratatime_speed_100 : std_logic_vector(1 downto 0);

	signal recv_ratatime_clear_100_1 : std_logic := '0';
	signal recv_ratatime_sync_100_1 : std_logic_vector(2 downto 0) := (others => '0');
	signal recv_ratatime_sync_lost_100_1 : std_logic_vector(1 downto 0) := (others => '0');
	signal recv_ratatime_bad_bits_100_1 : std_logic_vector(3 downto 0) := (others => '0');
	signal recv_ratatime_ticks_100_1 : std_logic_vector(63 downto 0);
	signal recv_ratatime_aux_bits_100_1 : std_logic_vector(14 downto 0);
	signal recv_ratatime_aux_sigs_100_1 : std_logic_vector(6 downto 0);
	signal recv_ratatime_speed_100_1 : std_logic_vector(1 downto 0);

	signal recv_ratatime_clear_125 : std_logic := '0';
	signal recv_ratatime_sync_125 : std_logic_vector(2 downto 0) := (others => '0');
	signal recv_ratatime_sync_lost_125 : std_logic_vector(1 downto 0) := (others => '0');
	signal recv_ratatime_bad_bits_125 : std_logic_vector(3 downto 0) := (others => '0');
	signal recv_ratatime_ticks_125 : std_logic_vector(63 downto 0);
	signal recv_ratatime_aux_bits_125 : std_logic_vector(14 downto 0);
	signal recv_ratatime_aux_sigs_125 : std_logic_vector(6 downto 0);
	signal recv_ratatime_speed_125 : std_logic_vector(1 downto 0);

	-- LED signals
	signal leds : std_logic_vector(7 downto 0) := (others => '0');
	signal display_leds : std_logic_vector(2 downto 0) := (others => '0');
	signal display_leds_pp1 : std_logic_vector(2 downto 0) := (others => '0');
	signal display_leds_pp2 : std_logic_vector(2 downto 0) := (others => '0');

        signal leds_00_pp0 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_00_pp1 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_00_pp2 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_01_pp0 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_01_pp1 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_01_pp2 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_10_pp0 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_10_pp1 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_10_pp2 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_11_pp0 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_11_pp1 : std_logic_vector(7 downto 0) := (others => '0');
        signal leds_11_pp2 : std_logic_vector(7 downto 0) := (others => '0');

	-- UART TX ---
	signal uart_tx_trg    : std_logic := '0';
	signal uart_tx_byte   : std_logic_vector(7 downto 0) := (others => '0');
	signal uart_tx_active : std_logic := '0';
	signal uart_tx_out    : std_logic := '0';
	signal uart_tx_done   : std_logic := '0';

	constant send_string : string := "Exploder";
	type arr_type is array(7 downto 0) of std_logic_vector(7 downto 0);
	signal send_chars : arr_type;
	signal char_select : integer := 0;
	signal char_select_logic : std_logic_vector(3 downto 0) := (others => '0');

begin

	char_select_logic <= std_logic_vector(to_unsigned(char_select, 4));

	send_chars(0) <= std_logic_vector(to_unsigned(character'pos(send_string(1)), 8));
	send_chars(1) <= std_logic_vector(to_unsigned(character'pos(send_string(2)), 8));
	send_chars(2) <= std_logic_vector(to_unsigned(character'pos(send_string(3)), 8));
	send_chars(3) <= std_logic_vector(to_unsigned(character'pos(send_string(4)), 8));
	send_chars(4) <= std_logic_vector(to_unsigned(character'pos(send_string(5)), 8));
	send_chars(5) <= std_logic_vector(to_unsigned(character'pos(send_string(6)), 8));
	send_chars(6) <= std_logic_vector(to_unsigned(character'pos(send_string(7)), 8));
	send_chars(7) <= std_logic_vector(to_unsigned(character'pos(send_string(8)), 8));

    -- UART TX
	uart_output : if use_uart_output = true generate
	uart_tx : entity work.UART_TX
	generic map (
		CLKS_PER_BIT => 1085
	)
	port map (
		i_Clock => clk_125,
		i_Tx_DV => uart_tx_trg,
		i_Tx_Byte => uart_tx_byte,
		o_Tx_Active => uart_tx_active,
		o_Tx_Serial => uart_tx_out,
		o_Tx_Done => uart_tx_done
	);
	end generate;

	-- Clocks
	clock_generators : entity work.clock_generators
	port map (
		-- internal oscillator:
	    --   -> clk_125_int internal 125 MHz
	    --   -> clk_100     internal 100 MHz
	    --   -> clk_200_int internal 200 MHz
		clk_125p_i		=> ck_125p_in,
		clk_125n_i		=> ck_125n_in,
		clk_125_o		=> clk_125_int,
		clk_100_o		=> clk_100,
		clk_200_o		=> clk_200_int,

		-- rataclock input:
		--   -> clk_125_rata derived clock from rataclock signal
		clk_25_i		  => rataclock_in,
		clk_125_rata_o	  => clk_125_rata,
		clk_125_90_rata_o => open,

		-- butis input:
		--   -> clk_200_butis derived 200 MHz
		--   -> clk_125_butis derived 125 MHz
		--   -> clk_10_butis  derived  10 MHz
		clk_200_i		=> clk_200_butis_buf,
		clk_200_butis_o	=> clk_200_butis,
		clk_125_butis_o => clk_125_butis,
		clk_10_butis_o  => clk_10_butis,

		-- multiplexer input
		--   -> clk_125    passed through
		--   -> clk_125_90 phase shifted for rataclock receiver
		clk_125_mux_i    => clk_125_mux,
		clk_125_mux_o    => clk_125,
		clk_125_90_mux_o => clk_125_90,

		locked_25_o	    => locked_25,
		locked_100_o	=> locked_100,
		locked_125_o	=> locked_125,
		locked_200_butis_o	=> locked_200_butis,

		stopped_200_butis_o => stopped_200_butis
	);

	butis_in : IBUFG port map (
		o => clk_200_butis_buf,
		i => any_input(0)
	);
	butis_time <= butis_time_in;

	uart_tx_buffer : OBUF port map (
		o => res_out,
		i => uart_tx_out
	);

	-- Serial UART sender for debug info
	uart_sender : process(clk_125)
	begin
		if (rising_edge(clk_125)) then
			if (uart_tx_active = '0') then
				uart_tx_trg <= '1';
			else
				uart_tx_trg <= '0';
			end if;
			if (uart_tx_done = '1') then
				if (char_select = 7) then
					char_select <= 0;
				else
					char_select <= char_select + 1;
				end if;
			end if;
		end if;
	end process;
	uart_tx_byte <= send_chars(char_select);

	-- Select internal or external 200 MHz clock via jumpers
	--           res bits    7      0
	--         silkscreen    8      1
	-- select_200_mhz_clk = [xxxxxxx:], use internal clock
	-- select_200_mhz_clk = [xxxxxxx|], use external butis clock (any input(0))
	-- clock_multiplexer_200 : entity work.clock_multiplexer2
	-- port map (
	-- 	clk_select => select_200_mhz_clk,
	-- 	clk_in0 => clk_200_int,
	-- 	clk_in1 => clk_200_butis,
	-- 	clk_out => clk_200
	-- 	);
	clk_200 <= clk_200_butis;

	-- Select duty cycle for rataclock sender
	--           res bits    7      0
	--         silkscreen    8      1
	-- select_rataclock_duty = [xxxxxxx:], duty cycle 2,3
	-- select_rataclock_duty = [xxxxxxx|], duty cycle 3,4
	process(select_rataclock_duty)
	begin
		case not select_rataclock_duty is
			when '0' =>
					send_low_min_clks <= "000010";
					send_low_max_clks <= "000011";
			when '1' =>
					send_low_min_clks <= "000011";
					send_low_max_clks <= "000100";
		end case;
	end process;


	-- Select clock source for 125 MHz clock via jumpers
	--           res bits    7      0
	--         silkscreen    8      1
	-- select_125_mhz_clk = [xxxxx::x], use internal clock
	-- select_125_mhz_clk = [xxxxx:|x], use clock from rataclock input		
	-- select_125_mhz_clk = [xxxxx|:x], use clock from butis input
	-- select_125_mhz_clk = [xxxxx||x], no clock		
	clock_multiplexer_125 : entity work.clock_multiplexer4
	port map (
		-- warning: res bits are inverted!
		clk_select => not select_125_mhz_clk,
		clk_in0 => clk_125_int,
		clk_in1 => clk_125_rata,
		clk_in2 => clk_125_butis,
		clk_in3 => '0',
		clk_out => clk_125_mux
		);

	--           res bits    7      0
	--         silkscreen    8      1
	-- select_led_display = [xx::xxxx], show clock + rataclock/butis sync status
	-- select_led_display = [xx:|xxxx], show PLL/DCM locks
	-- select_led_display = [xx|:xxxx], show ratatime sync status
	-- select_led_display = [xx||xxxx], show timestamp
			-- warning: res bits are inverted!
	leds_11_pp0 <=	butis_count(26)
			& recv_butis_synced
			& recv_butis_error
			& recv_sync_status_out(1 downto 0)
			& recv_sync_lost_out(2 downto 1)
			& recv_butis_timestamp_lo(30);

	leds_10_pp0 <=	locked_25
			& locked_100
			& locked_125
			& locked_200_butis
			& "0000";

	leds_01_pp0 <=	recv_ratatime_sync(2 downto 0)
			& recv_ratatime_sync_lost(1 downto 0)
			& recv_ratatime_speed(1 downto 0)
			& recv_ratatime_aux_sigs(0);

	leds_00_pp0 <=	std_logic_vector(tick_ns(30 downto 29))
			& std_logic_vector(tick_ns_recv_pp2(30 downto 29))
			& recv_butis_timestamp_lo(30 downto 29)
			& recv_ratatime_timestamp(30 downto 29);

        process(clk_125)
        begin
          if (rising_edge(clk_125)) then
            leds_11_pp1 <= leds_11_pp0;  leds_11_pp2 <= leds_11_pp1;
            leds_10_pp1 <= leds_10_pp0;  leds_10_pp2 <= leds_10_pp1;
            leds_01_pp1 <= leds_01_pp0;  leds_01_pp2 <= leds_01_pp1;
            leds_00_pp1 <= leds_00_pp0;  leds_00_pp2 <= leds_00_pp1;

            case select_led_display is
              when "11" => leds <= leds_11_pp2;
              when "10" => leds <= leds_10_pp2;
              when "01" => leds <= leds_01_pp2;
              when "00" => leds <= leds_00_pp2;
			  when others => leds <= leds_00_pp2;
            end case;
          end if;
        end process;

        process(clk_125)
        begin
          if (rising_edge(clk_125)) then
            tick_ns_pp1                 <= tick_ns;
            tick_ns_pp2                 <= tick_ns_pp1;

            tick_ns_recv_pp1            <= tick_ns_recv;
            tick_ns_recv_pp2            <= tick_ns_recv_pp1;
            tick_ns_recv_pp3            <= tick_ns_recv_pp2;
            tick_ns_recv_pp4            <= tick_ns_recv_pp3;

            recv_butis_timestamp_pp1    <= recv_butis_timestamp;
            recv_butis_timestamp_pp2    <= recv_butis_timestamp_pp1;

            recv_ratatime_timestamp_pp1 <= recv_ratatime_timestamp;
            recv_ratatime_timestamp_pp2 <= recv_ratatime_timestamp_pp1;

            timestamp_ns_xx1            <= timestamp_ns_xx2;
            timestamp_ns                <= timestamp_ns_xx1;
          end if;
        end process;

   -- Select timestamp source
	--                res bits    7      0
	--              silkscreen    8      1
	-- select_timestamp_source = [::xxxx], rataclock sender (internal)
	-- select_timestamp_source = [:|xxxx], rataclock receiver (external)
	-- select_timestamp_source = [|:xxxx], butis receiver (external)
	-- select_timestamp_source = [||xxxx], ratatime receiver (external)
	with select_timestamp_source select timestamp_ns_xx2 <=
		-- warning: res bits are inverted!
		tick_ns_pp2 when "11",
		tick_ns_recv_pp4 when "10",
		unsigned(recv_butis_timestamp_pp2) when "01",
		unsigned(recv_ratatime_timestamp_pp2) when "00";

	-- send rataclock signal to outputs
	send_rataclock_forward_lemo_naive : ODDR2
	port map (
		d0 => send_rataclock_lemo_out,
		d1 => send_rataclock_lemo_out,
		c0 => send_clk,
		c1 => not send_clk,
		q => send_rataclock_oddr_lemo_naive
	);

	--   this is only better for the broken I/O board
	d0_prepare <= send_rataclock_lemo_out or
				  (send_trail_short and '1') or
				  (send_trail_long and '0');
	d1_prepare <= send_rataclock_lemo_out;

	send_rataclock_forward_lemo_trail : ODDR2
	port map (
		d0 => d0_prepare,
		d1 => d1_prepare,
		c0 => send_clk,
		c1 => not send_clk,
		q => send_rataclock_oddr_lemo_trail
	);

	send_rataclock_forward_lvds_naive : ODDR2
	port map (
		d0 => send_rataclock_lvds_out,
		d1 => send_rataclock_lvds_out,
		c0 => send_clk,
		c1 => not send_clk,
		q => send_rataclock_oddr_lvds_naive
	);

	send_rataclock_forward_ecl : ODDR2
	port map (
		d0 => send_rataclock_ecl_out,
		d1 => send_rataclock_ecl_out,
		c0 => send_clk,
		c1 => not send_clk,
		q => send_rataclock_oddr_ecl
	);

	-- send ratatime signal to outputs
	send_ratatime_forward_lemo_naive : ODDR2
	port map (
		d0 => send_ratatime,
		d1 => send_ratatime,
		c0 => clk_100,
		c1 => not clk_100,
		q => send_ratatime_oddr_lemo
	);

	-- send butis clock and time to LVDS for loopback
	clk_200_forward_lvds : ODDR2
	port map (
		d0 => '1',
		d1 => '0',
		c0 => clk_200_int,
		c1 => not clk_200_int,
		q => clk_200_to_lvds
	);

	-- the sender is running on the internal 200 MHz butis clock
	-- it will *not* repeat whatever comes on the input!
	clk_200_send_butis_time_lvds : ODDR2
	port map (
		d0 => send_butis_time,
		d1 => send_butis_time,
		c0 => clk_200_int,
		c1 => not clk_200_int,
		q => send_butis_time_oddr_lvds
	);

	-- I/O buffers
	io_buffers : entity work.io_buffers
	generic map (
		any_in_width => any_in_width,
		ecl_out_width => ecl_out_width,
		lvds_out_width => lvds_out_width,
		lemo_out_width => lemo_out_width,
		lemo_in_width => lemo_in_width
	)
	port map (
	    -- OUTPUTS --
		lemo_out_i(0) => send_rataclock_oddr_lemo_naive,
		lemo_out_i(1) => send_rataclock_slow,
		lemo_out_i(2) => send_rataclock_oddr_lemo_trail,
		lemo_out_i(3) => send_ratatime_oddr_lemo,
		lemo_out_i(4) => send_butis_t0_signal,
		lemo_out_i(5) => recv_butis_t0,
		lemo_out_i(6) => '0',
		lemo_out_i(7) => '1',
		ecl_out_i(0) => send_rataclock_slow,
		ecl_out_i(1) => send_ratatime,
		ecl_out_i(2) => send_butis_t0_signal,
		ecl_out_i(3) => send_rataclock_oddr_ecl,
		ecl_out_i(7 downto 4) => "0000",
		lvds_out_i(0) => clk_200_to_lvds,
		lvds_out_i(1) => send_butis_time_oddr_lvds,
		lvds_out_i(2) => send_rataclock_oddr_lvds_naive,
		lvds_out_i(3) => send_ratatime,
		lvds_out_i(4) => send_butis_t0_signal,
		lvds_out_i(5) => send_rataclock_sync,
		lvds_out_i(7 downto 6) => "00",
		-- INPUTS --
		lemo_in_o(0) => rataclock_in,
		lemo_in_o(1) => recv_clear_status,
		lemo_in_o(2) => recv_ratatime,
		lemo_in_o(3) => recv_ratatime_clear,
		lemo_in_o(7 downto 4) => open,
        res_o => res_pp0,
		-- any_in_o(0)		 => butis_clock, -- handled below
		any_in_o(1)          => butis_time_in,
		any_in_o(5 downto 2) => open,
		any_in_o(6)          => open,
		hpv_o => hpv,
		led_i => leds,
		any_in_i => any_input(6 downto 1),
		ecl_out_o => ecl_output,
		lvds_out_o => lvds_output,
		lemo_out_o => lemo_output,
		lemo_in_i => lemo_input,
		res_i => res,
		display_led_i => display_leds_pp2,
		display_led_o => display_led_out,
		select_out_o => select_out
	);

	display_leds(1) <= recv_sync_status_out(2);

        process(clk_125)
        begin
          if (rising_edge(clk_125)) then
            display_leds_pp1 <= display_leds;
            display_leds_pp2 <= display_leds_pp1;
          end if;
        end process;

        process(clk_125)
        begin
          if (rising_edge(clk_125)) then
            res_pp1 <= res_pp0;
            res_pp2 <= res_pp1;
          end if;
        end process;

        select_rataclock_duty      <= res_pp2(0);
        select_125_mhz_clk         <= res_pp2(2 downto 1);
        select_led_display         <= res_pp2(4 downto 3);
        select_timestamp_source    <= res_pp2(6 downto 5);

	-- Time generator
	time_generator : process(clk_125)
	begin
		if rising_edge(clk_125) then
			tick_ns <= tick_ns + 8;
		end if;
	end process;

	butis_sender_gen : if use_butis_sender = true generate

	-- Butis input clock counter (downscaled to 10 MHz)
	butis_counter : process(clk_10_butis)
	begin
		if rising_edge(clk_10_butis) then
			butis_count <= butis_count + 20;
		end if;
	end process;

	-- Butis T0 (Schakel protocol) sender (for local loopback)
	-- the sender runs on the internal 200 MHz clock.
	-- as such it is always undisciplined and should be used only for
	-- debug purposes.
	butis_sender : entity work.butis_t0_proto_sender
	port map(
		butis_clk	=> clk_200_int,
		i_butis_t0 	=> send_butis_mark100kHz,
		i_timestamp => send_butis_timestamp_next_t0,
		o_serial 	=> send_butis_time_xx6
	);

		-- pipeline to mitigate timing errors
        process (clk_200_int)
        begin
          if (rising_edge(clk_200_int)) then
            send_butis_time_xx5 <= send_butis_time_xx6;
            send_butis_time_xx4 <= send_butis_time_xx5;
            send_butis_time_xx3 <= send_butis_time_xx4;
            send_butis_time_xx2 <= send_butis_time_xx3;
            send_butis_time_xx1 <= send_butis_time_xx2;
            send_butis_time     <= send_butis_time_xx1;
          end if;
        end process;

	butis_sender_input : process (clk_200_int)
	begin
		if (rising_edge(clk_200_int)) then

			send_butis_timestamp <= std_logic_vector(
									unsigned(send_butis_timestamp) + 5);
			send_butis_cycles <= send_butis_cycles - 1;
			send_butis_mark100kHz <= '0';

			if send_butis_cycles = 1 then
				send_butis_cycles <= 2000;
				send_butis_mark100kHz <= '1';
 			    send_butis_timestamp_next_t0_pp1 <= std_logic_vector(
												unsigned(send_butis_timestamp)
												+ 10000);
				send_butis_timestamp_next_t0_pp2 <= send_butis_timestamp_next_t0_pp1;
				send_butis_timestamp_next_t0     <= send_butis_timestamp_next_t0_pp2;
				send_butis_t0_signal <= '1';
			else
				if (send_butis_cycles < 10) then
					send_butis_t0_signal <= '1';
				else
					send_butis_t0_signal <= '0';
				end if;
		   end if;
		end if;
	end process;

	end generate;

	-- Butis T0 (Schakel protocol) receiver
	butis_receiver_gen : if use_butis_receiver = true generate

	recv_butis_clock <= clk_200;

	butis_receiver : entity work.butis_t0_proto_receiver
	port map(
		clk         => recv_butis_clock,
		i_serial    => butis_time,
		o_butis_t0  => recv_butis_t0,
		o_timestamp_lo => recv_butis_timestamp_lo,
		o_timestamp_hi => recv_butis_timestamp_hi,
		o_synced    => recv_butis_synced,
		o_error     => recv_butis_error
    );
	end generate;

	-- Rataclock sender (free running)
	send_clock : entity work.rataser_clock_send
	generic map(period_bits => 6)
	port map(
		clk                => send_clk,

		tick_ns            => std_logic_vector(timestamp_ns),
		aux_sigs           => send_aux_sigs,
		info_bit           => send_info_bit,

		pulse_period_clks  => "000101", -- 5, 25 MHz output pulse
		duty_low_min_clks  => send_low_min_clks, -- "000011", -- 3
		duty_low_max_clks  => send_low_max_clks, -- "000100", -- 4
		eight_slot         => '1',

		pulse_period_ns    => "0000101000", -- 40
		use_pulse_period_ns => '1',

		message_delay_ns   => (others => '0'),

		transmit           => send_rataclock_xx1,
		transmit_sync      => send_rataclock_sync,

		trail_short        => send_trail_short,
		trail_long         => send_trail_long
	);

	-- pipeline send_rataclock to mitigate timing errors
	-- need to pipeline separately for different outputs,
	-- because they are on different sides of the die
	process (send_clk)
	begin
		if (rising_edge(send_clk)) then
			send_rataclock_lvds_xx2 <= send_rataclock_xx1;
			send_rataclock_lvds_xx3 <= send_rataclock_lvds_xx2;
			send_rataclock_lvds_xx4 <= send_rataclock_lvds_xx3;
			send_rataclock_lvds_xx5 <= send_rataclock_lvds_xx4;
			send_rataclock_lvds_out <= send_rataclock_lvds_xx5;

			send_rataclock_lemo_xx2 <= send_rataclock_xx1;
			send_rataclock_lemo_xx3 <= send_rataclock_lemo_xx2;
			send_rataclock_lemo_xx4 <= send_rataclock_lemo_xx3;
			send_rataclock_lemo_xx5 <= send_rataclock_lemo_xx4;
			send_rataclock_lemo_xx6 <= send_rataclock_lemo_xx5;
			send_rataclock_lemo_xx7 <= send_rataclock_lemo_xx6;
			send_rataclock_lemo_xx8 <= send_rataclock_lemo_xx7;
			send_rataclock_lemo_out <= send_rataclock_lemo_xx8;

			send_rataclock_ecl_xx2  <= send_rataclock_xx1;
			send_rataclock_ecl_xx3  <= send_rataclock_ecl_xx2;
			send_rataclock_ecl_xx4  <= send_rataclock_ecl_xx3;
			send_rataclock_ecl_xx5  <= send_rataclock_ecl_xx4;
			send_rataclock_ecl_xx6  <= send_rataclock_ecl_xx5;
			send_rataclock_ecl_xx7  <= send_rataclock_ecl_xx6;
			send_rataclock_ecl_xx8  <= send_rataclock_ecl_xx7;
			send_rataclock_ecl_xx9  <= send_rataclock_ecl_xx8;
			send_rataclock_ecl_xxA  <= send_rataclock_ecl_xx9;
			send_rataclock_ecl_out  <= send_rataclock_ecl_xxA;
		end if;
	end process;

	-- Slow Rataclock sender (free running, no eight_slot)
	send_clock_slow : entity work.rataser_clock_send
	generic map(period_bits => 6)
	port map(
		clk                => send_clk,

		tick_ns            => std_logic_vector(timestamp_ns),
		aux_sigs           => send_aux_sigs,
		info_bit           => send_info_bit,

		pulse_period_clks  => "100000", -- 32, 3.9 MHz output pulse
		duty_low_max_clks  => "010000", -- 16
		duty_low_min_clks  => "001000", --  8
		eight_slot         => '0',

		pulse_period_ns    => "0100000000", -- 256 ns
		use_pulse_period_ns => '1',

		message_delay_ns   => (others => '0'),

		transmit           => send_rataclock_slow,
		transmit_sync      => send_rataclock_slow_sync,

		trail_short        => open,
		trail_long         => open
	);

	-- Rataclock receiver
	rataclock_receiver : if use_rataclock_receiver = true generate
	recv_clock : entity work.rataser_clock_recv
    generic map(period_bits => 6,
                num_last_edges => 4,
				max_skew_4phase_sample => "1.5 ns")
    port map(
      clk               => recv_clk,
      clk90             => recv_clk90,

      receive           => recv_rataclock,
      eight_slot        => '1',

      expect_edge       => recv_expect_edge,
      last_edges        => recv_last_edges,
      use_auto_edge     => '1',
      auto_edge         => recv_auto_edge,

      receive_delay_ns  => (others => '0'),

      tick_ns_lo        => recv_tick_ns_lo,
      tick_ns_hi        => recv_tick_ns_hi,
      aux_sigs          => recv_aux_sigs,
      info_bit          => recv_info_bit,
      msg_strobe        => recv_msg_strobe,

      sync_status       => recv_sync_status_xx1,
      sync_lost         => recv_sync_lost_xx1,
      bad_signals       => open,
      clear_status      => recv_clear_status,

      pulse_period_clks => "000101", -- 5
      clk_period_ns     => "0000001000" -- 8 ns
      );
	end generate;

	process (clk_125)
	begin
		if (rising_edge(clk_125)) then
			recv_sync_lost_xx2 <= recv_sync_lost_xx1;
			recv_sync_lost_xx3 <= recv_sync_lost_xx2;
			recv_sync_lost_out <= recv_sync_lost_xx3;

			recv_sync_status_xx2 <= recv_sync_status_xx1;
			recv_sync_status_xx3 <= recv_sync_status_xx2;
			recv_sync_status_out <= recv_sync_status_xx3;
		end if;
	end process;

	process(clk_100)
	begin
		if (rising_edge(clk_100)) then
			timestamp_ns_100 <= timestamp_ns;
		end if;
	end process;

	-- ratatime sender
	sender_ratatime_block : if use_ratatime_sender = true generate
	sender_ratatime : entity work.serial_timestamp_sender
	port map (
		clk => clk_100,
		ticks => std_logic_vector("0000" & timestamp_ns_100(63 downto 4)),
		aux_bits => (others => '0'),
		aux_sigs => (others => '0'),
		speed => "11",
		transmit => send_ratatime,
		transmit_sync => send_ratatime_sync
	);
	end generate;

	-- ratatime receiver
	ratatime_receiver : if use_ratatime_receiver = true generate
	process(clk_100)
	begin
		if (rising_edge(clk_100)) then
			clk_100_next_counter <= std_logic_vector(unsigned(clk_100_counter) + 1);
			clk_100_counter <= clk_100_next_counter;
			recv_ratatime_new_cycle_pulse <=
				clk_100_counter(13) xor
				clk_100_next_counter(13);
		end if;
	end process;

	receiver_ratatime : entity work.serial_timestamp_decoder
	port map (
		clk                => clk_100,
		receive            => recv_ratatime,
		latch              => '0',
		new_cycle_estimate => recv_ratatime_new_cycle_pulse,
		clear_failure      => recv_ratatime_clear_100,
		bitstr_synced      => recv_ratatime_sync_100(2),
		ptn_synced         => recv_ratatime_sync_100(1),
		time_locked        => recv_ratatime_sync_100(0),
		bitstr_sync_lost   => recv_ratatime_sync_lost_100(1),
		ptn_sync_lost      => recv_ratatime_sync_lost_100(0),
		bad_bits           => recv_ratatime_bad_bits_100,
		ticks              => recv_ratatime_ticks_100,
		aux_bits           => recv_ratatime_aux_bits_100,
		aux_sigs           => recv_ratatime_aux_sigs_100,
		recv_speed         => recv_ratatime_speed_100
	);
	end generate;

	process(clk_100)
	begin
          if (rising_edge(clk_100)) then
            recv_ratatime_sync_100_1      <= recv_ratatime_sync_100;
            recv_ratatime_sync_lost_100_1 <= recv_ratatime_sync_lost_100;
            recv_ratatime_bad_bits_100_1  <= recv_ratatime_bad_bits_100;
            recv_ratatime_ticks_100_1     <= recv_ratatime_ticks_100;
            recv_ratatime_aux_bits_100_1  <= recv_ratatime_aux_bits_100;
            recv_ratatime_aux_sigs_100_1  <= recv_ratatime_aux_sigs_100;
            recv_ratatime_speed_100_1     <= recv_ratatime_speed_100;

            recv_ratatime_clear_100_1     <= recv_ratatime_clear;
            recv_ratatime_clear_100       <= recv_ratatime_clear_100_1;
          end if;
        end process;

	process(clk_125)
	begin
          if (rising_edge(clk_125)) then
            recv_ratatime_clear_125     <= recv_ratatime_clear_100;
            recv_ratatime_sync_125      <= recv_ratatime_sync_100_1;
            recv_ratatime_sync_lost_125 <= recv_ratatime_sync_lost_100_1;
            recv_ratatime_bad_bits_125  <= recv_ratatime_bad_bits_100_1;
            recv_ratatime_ticks_125     <= recv_ratatime_ticks_100_1;
            recv_ratatime_aux_bits_125  <= recv_ratatime_aux_bits_100_1;
            recv_ratatime_aux_sigs_125  <= recv_ratatime_aux_sigs_100_1;
            recv_ratatime_speed_125     <= recv_ratatime_speed_100_1;

            recv_ratatime_sync      <= recv_ratatime_sync_125;
            recv_ratatime_sync_lost <= recv_ratatime_sync_lost_125;
            recv_ratatime_bad_bits  <= recv_ratatime_bad_bits_125;
            recv_ratatime_ticks     <= recv_ratatime_ticks_125;
            recv_ratatime_aux_bits  <= recv_ratatime_aux_bits_125;
            recv_ratatime_aux_sigs  <= recv_ratatime_aux_sigs_125;
            recv_ratatime_speed     <= recv_ratatime_speed_125;
          end if;
        end process;

	-- Connections
	send_clk <= clk_125;
	send_aux_sigs <= (others => '0');
	send_info_bit <= '1';

	recv_clk <= clk_125;      -- use clock derived from ext. rataclock
	recv_clk90 <= clk_125_90; -- use clock derived from ext. rataclock
	recv_rataclock <= rataclock_in;   -- use external input

	tick_ns_recv(63 downto 32) <= unsigned(recv_tick_ns_hi);
	tick_ns_recv(31 downto 0) <= unsigned(recv_tick_ns_lo);
	recv_butis_timestamp(63 downto 32) <= recv_butis_timestamp_hi;
	recv_butis_timestamp(31 downto 0) <= recv_butis_timestamp_lo;

end behavior;

-- vim: sw=4 ts=4
